import argparse
from pathlib import Path
from typing import List

import fq_utils
import runner
from runner import ProcessCmd


def execute_fastqc(dragen_args: List[str], fastqs: List[str], multiqc: str):
    static_cmd = runner.create_dragen_cmd(dragen_args, with_ref=False)
    samples, fastq_list = fq_utils.process_fastqs(fastqs, Path('.'))
    cmds: List[ProcessCmd] = [ProcessCmd('/opt/edico/bin/dragen --partial-reconfig HMM --ignore-version-check true')]
    for sample_name in samples:
        cmds.append(ProcessCmd(' '.join([
            static_cmd,
            f'--output-file-prefix {sample_name}',
            f'--fastq-list {fastq_list}',
            f'--fastq-list-sample-id {sample_name}',
        ])))

    if multiqc.lower() == 'all':
        cmds.append(ProcessCmd('multiqc -o . .'))
    elif multiqc.lower() == 'none':
        pass
    else:
        raise ValueError('MultiQC modes other than "all" are not currently supported')

    runner.run_dragen_cmds(cmds)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--fastqs', nargs='*', default=[], type=str)
    parser.add_argument('--multiqc', type=str)
    parser.add_argument('--debug', action='store_true')

    args, extras = parser.parse_known_args()

    runner.execute_with_no_traceback(
        lambda: execute_fastqc(extras, args.fastqs, args.multiqc),
        args.debug
    )


if __name__ == '__main__':
    main()
