import os
import pathlib
import shlex
import subprocess
import sys
from dataclasses import dataclass
from pathlib import Path
from typing import List, Callable, Any

DRAGEN_LIC_ENV_VAR = "SECURE_DRAGEN_CREDS"


@dataclass
class ProcessCmd:
    cmd: str
    working_dir: str = None
    check: bool = True


def run_dragen_cmds(process_cmds: List[ProcessCmd]):
    for process_cmd in process_cmds:
        if process_cmd.working_dir:
            os.makedirs(process_cmd.working_dir, exist_ok=True)

        try:
            subprocess.run(shlex.split(process_cmd.cmd), check=process_cmd.check)
        except subprocess.CalledProcessError as e:
            if process_cmd.cmd.startswith('/opt/edico/bin/dragen'):
                # Copy /var/log/dragen to output
                # Need to first remove the symlink
                pathlib.Path('/var/log/dragen/dragen_last_good_run.log').unlink(missing_ok=True)
                if process_cmd.working_dir:
                    subprocess.run(f'cp -r /var/log/dragen {process_cmd.working_dir}'.split())
                else:
                    os.makedirs('dragen-logs', exist_ok=True)
                    subprocess.run(f'cp -r /var/log/dragen dragen-logs/'.split())

            raise e


def create_dragen_cmd(static_args: List[str], with_ref: bool = True) -> str:
    output_dir = Path('.')  # IAP cannot glob on directory names so just output everything to the root dir

    cmd = ' '.join([
        f'/opt/edico/bin/dragen --lic-instance-id-location /opt/instance-identity',
        f'--intermediate-results-dir /ephemeral',
        f'--output-directory {str(output_dir)}',
        *static_args
    ])
    if os.getenv(DRAGEN_LIC_ENV_VAR):
        cmd = ' '.join([cmd, f'--lic-server {os.getenv(DRAGEN_LIC_ENV_VAR)}'])
    if with_ref is True:
        cmd = ' '.join([cmd, f'--ref-dir {REF_DIR}', ])
    return cmd


def create_setup_ref_cmds(ref_tar: str) -> List[ProcessCmd]:
    return [
        ProcessCmd('/opt/edico/bin/dragen --partial-reconfig HMM --ignore-version-check true'),
        ProcessCmd(f'tar -C {REF_DIR} -xf {ref_tar}', REF_DIR),
    ]


def execute_with_no_traceback(func: Callable[[], Any], debug: bool) -> None:
    try:
        func()
    except Exception:
        if debug:
            raise
        else:
            sys.exit(1)


REF_DIR = '/ephemeral/reference/index'
